'use client'

import { useState } from 'react'
import Link from 'next/link'
import { Button } from "@/components/ui/button"
import { Card, CardContent } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { ChevronLeft, ChevronRight, Plus, FileText, Home } from 'lucide-react'
import {useRouter} from "next/navigation";

type PropertyType = 'apartment' | 'house' | 'villa' | 'office' | 'commercial'

type Property = {
  id: string
  name: string
  address: string
  type: PropertyType
  surface: number
  rooms: number
  documentsIncomplete: boolean
}

const propertyEmojis: Record<PropertyType, string> = {
  apartment: '🏢',
  house: '🏠',
  villa: '🏡',
  office: '🏢',
  commercial: '🏬',
}

const properties: Property[] = [
  { id: '1', name: 'Appartement Haussmannien', address: '15 Avenue des Champs-Élysées, 75008 Paris', type: 'apartment', surface: 120, rooms: 4, documentsIncomplete: true },
  { id: '2', name: 'Villa Côte d\'Azur', address: '7 Promenade des Anglais, 06000 Nice', type: 'villa', surface: 200, rooms: 6, documentsIncomplete: false },
  { id: '3', name: 'Bureau Moderne', address: '23 Rue de la République, 69002 Lyon', type: 'office', surface: 80, rooms: 3, documentsIncomplete: true },
  { id: '4', name: 'Local Commercial', address: '5 Rue du Commerce, 44000 Nantes', type: 'commercial', surface: 150, rooms: 2, documentsIncomplete: false },
]

export function BlockPage() {
  const [selectedProperty, setSelectedProperty] = useState<Property | null>(null)
const router= useRouter()
  const handleBack = () => {
    setSelectedProperty(null)
  }
  const handleReturnHome = () => {
    router.push('/properties/new')
  }

  if (selectedProperty) {
    return (
      <div className="min-h-screen bg-gray-50 p-4">
        <div className="max-w-3xl mx-auto">
          <div className="flex items-center justify-between mb-6">
            <Button 
              variant="ghost" 
              size="icon" 
              className="text-gray-600 hover:text-gray-800 rounded-lg"
              onClick={handleBack}
            >
              <ChevronLeft className="h-6 w-6" />
              <span className="sr-only">Retour</span>
            </Button>
            <h1 className="text-2xl font-bold text-gray-800">Gestion de la propriété</h1>
            <div className="w-10" />
          </div>

          <Card className="mb-6">
            <CardContent className="p-6">
              <div className="flex items-center space-x-4">
                <span className="text-4xl" role="img" aria-label={selectedProperty.type}>
                  {propertyEmojis[selectedProperty.type]}
                </span>
                <div>
                  <h2 className="text-xl font-semibold">{selectedProperty.name}</h2>
                  <p className="text-gray-500">{selectedProperty.address}</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <div className="grid grid-cols-2 gap-4">
            <Link href={`/settings/properties/${selectedProperty.id}/general`} passHref>
              <Card className="hover:shadow-md transition-shadow duration-200 cursor-pointer">
                <CardContent className="p-6 flex items-center justify-between">
                  <div className="flex items-center">
                    <Home className="h-6 w-6 mr-3 text-purple-600" />
                    <span className="font-medium">Général</span>
                  </div>
                  <ChevronRight className="h-5 w-5 text-gray-400" />
                </CardContent>
              </Card>
            </Link>
            <Link href={`/settings/properties/${selectedProperty.id}/documents`} passHref>
              <Card className="hover:shadow-md transition-shadow duration-200 cursor-pointer">
                <CardContent className="p-6 flex items-center justify-between">
                  <div className="flex items-center">
                    <FileText className="h-6 w-6 mr-3 text-purple-600" />
                    <span className="font-medium">Documents</span>
                    {selectedProperty.documentsIncomplete && (
                      <Badge variant="destructive" className="ml-2">
                        !
                      </Badge>
                    )}
                  </div>
                  <ChevronRight className="h-5 w-5 text-gray-400" />
                </CardContent>
              </Card>
            </Link>
          </div>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50 p-4">
      <div className="max-w-2xl mx-auto">
        <div className="flex items-center justify-between mb-6">
          <Button variant="ghost" size="icon" className="text-gray-600 hover:text-gray-800 rounded-lg">
            <ChevronLeft className="h-6 w-6" />
            <span className="sr-only">Retour</span>
          </Button>
          <h1 className="text-2xl font-bold text-gray-800">Mes Propriétés</h1>
          <Button 
            variant="ghost" 
            size="icon" 
            className="bg-purple-100 hover:bg-purple-200 text-purple-600 hover:text-purple-700 rounded-lg"
            onClick={handleReturnHome}
          >
            <Plus className="h-6 w-6" />
            <span className="sr-only">Ajouter une propriété</span>
          </Button>
        </div>
        <div className="space-y-4">
          {properties.map((property) => (
            <Card 
              key={property.id} 
              className="hover:shadow-md transition-shadow duration-200 rounded-lg cursor-pointer"
              onClick={() => setSelectedProperty(property)}
            >
              <CardContent className="p-4">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-3">
                    <span className="text-2xl" role="img" aria-label={property.type}>
                      {propertyEmojis[property.type]}
                    </span>
                    <div>
                      <h2 className="font-semibold">{property.name}</h2>
                      <p className="text-sm text-gray-500">{property.address}</p>
                    </div>
                  </div>
                  <div className="flex items-center">
                    {property.documentsIncomplete && (
                      <Badge variant="destructive" className="mr-2">
                        !
                      </Badge>
                    )}
                    <ChevronRight className="h-5 w-5 text-gray-400" />
                  </div>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    </div>
  )
}