'use client'

import { useState } from 'react'
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Label } from "@/components/ui/label"
import { Switch } from "@/components/ui/switch"
import { Separator } from "@/components/ui/separator"
import { ChevronLeft, Bell, Mail, MessageCircle } from 'lucide-react'
import Link from 'next/link'

export function BlockPage() {
  const [notifications, setNotifications] = useState({
    pushEnabled: true,
    emailEnabled: false,
    whatsappEnabled: false,
    inspectionScheduled: true,
    inspectionReminder: true,
    inspectionCompleted: true,
    inspectionReport: true,
    paymentReceived: true,
    paymentReminder: true,
  })

  const handleToggle = (key: keyof typeof notifications) => {
    setNotifications(prev => ({ ...prev, [key]: !prev[key] }))
  }

  const notificationChannels = [
    { id: 'pushEnabled', name: 'Notifications push', description: 'Recevoir des notifications sur lappareil', icon: Bell },
    { id: 'emailEnabled', name: 'Notifications par email', description: 'Recevoir des notifications par email', icon: Mail },
    { id: 'whatsappEnabled', name: 'Notifications WhatsApp', description: 'Recevoir des notifications via WhatsApp', icon: MessageCircle },
  ]

  const notificationTypes = [
    { id: 'inspectionScheduled', name: 'Inspections planifiées', description: 'Nouvelles inspections programmées' },
    { id: 'inspectionReminder', name: 'Rappels d\'inspection', description: 'Rappels pour les inspections à venir' },
    { id: 'inspectionCompleted', name: 'Inspections terminées', description: 'Inspections marquées comme terminées' },
    { id: 'inspectionReport', name: 'Rapports d\'inspection', description: 'Nouveaux rapports d\'inspection disponibles' },
    { id: 'paymentReceived', name: 'Paiements reçus', description: 'Confirmation des paiements reçus' },
    { id: 'paymentReminder', name: 'Rappels de paiement', description: 'Rappels pour les paiements en attente' },
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      <div className="sticky top-0 z-10 bg-white p-4 shadow-sm">
        <div className="flex items-center justify-between max-w-md mx-auto">
          <Link href="/settings">
            <Button variant="ghost" size="icon" className="text-gray-600 hover:text-gray-900">
              <ChevronLeft className="h-6 w-6" />
              <span className="sr-only">Retour</span>
            </Button>
          </Link>
          <h1 className="text-xl font-bold text-gray-900">Notifications</h1>
          <div className="w-10"></div>
        </div>
      </div>

      <div className="p-4 max-w-md mx-auto space-y-6">
        <Card>
          <CardHeader>
            <CardTitle className="text-lg font-semibold text-gray-900">Canaux de notification</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {notificationChannels.map((channel) => (
              <div key={channel.id} className="flex items-center space-x-4">
                <channel.icon className="h-5 w-5 text-gray-500" />
                <div className="flex-grow">
                  <Label htmlFor={channel.id} className="flex flex-col">
                    <span className="font-medium text-gray-900">{channel.name}</span>
                    <span className="text-sm text-gray-600">{channel.description}</span>
                  </Label>
                </div>
                <Switch
                  id={channel.id}
                  checked={notifications[channel.id as keyof typeof notifications]}
                  onCheckedChange={() => handleToggle(channel.id as keyof typeof notifications)}
                  className="data-[state=checked]:bg-purple-600"
                />
              </div>
            ))}
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="text-lg font-semibold text-gray-900">Types de notifications</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {notificationTypes.map((type, index) => (
              <div key={type.id}>
                {index > 0 && <Separator className="my-4" />}
                <div className="flex items-center justify-between">
                  <Label htmlFor={type.id} className="flex flex-col">
                    <span className="font-medium text-gray-900">{type.name}</span>
                    <span className="text-sm text-gray-600">{type.description}</span>
                  </Label>
                  <Switch
                    id={type.id}
                    checked={notifications[type.id as keyof typeof notifications]}
                    onCheckedChange={() => handleToggle(type.id as keyof typeof notifications)}
                    className="data-[state=checked]:bg-purple-600"
                    disabled={!notifications.pushEnabled && !notifications.emailEnabled && !notifications.whatsappEnabled}
                  />
                </div>
              </div>
            ))}
          </CardContent>
        </Card>
      </div>
    </div>
  )
}